# 機能設計書 15-畳み込み演算（Convolution Operations）

## 概要

本ドキュメントは、TensorFlowにおける畳み込み演算（1D/2D/3D畳み込み、転置畳み込み、深さ方向畳み込み等）に関する設計書である。C++カーネル層（nn_ops.cc）とPython Kerasレイヤー層（convolutional.py）の両面から仕様を詳述する。

### 本機能の処理概要

畳み込み演算機能は、ニューラルネットワークの中核的な演算である畳み込み操作を提供する。入力テンソルに対して畳み込みカーネル（フィルタ）をスライディングウィンドウ方式で適用し、特徴マップを生成する。

**業務上の目的・背景**：畳み込みニューラルネットワーク（CNN）は画像認識、物体検出、自然言語処理などの分野で広く使用される。畳み込み演算は局所的な特徴の抽出を可能にし、重み共有によるパラメータ効率化と平行移動不変性を実現する。

**機能の利用シーン**：画像分類モデル（ResNet、VGG等）の構築、物体検出モデル（YOLO、SSD等）の特徴抽出、セマンティックセグメンテーション（U-Net等）のエンコーダ/デコーダ、テキスト分類の1D畳み込み、動画解析の3D畳み込み。

**主要な処理内容**：
1. Conv1D/Conv2D/Conv3D畳み込み演算の実行
2. 転置畳み込み（Conv2DTranspose等）によるアップサンプリング
3. 深さ方向畳み込み（DepthwiseConv2D）と点方向畳み込み（SeparableConv2D）
4. パディング処理（valid/same/causal）
5. ストライド・ディレーション制御
6. グループ畳み込み
7. データフォーマット（channels_last/channels_first）対応

**関連システム・外部連携**：cuDNN（GPU畳み込み最適化）、XNNPACK（モバイル最適化）、XLA（JITコンパイル最適化）。

**権限による制御**：特になし。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | - | - | 本機能はモデル内部で使用される計算処理であり、画面に直接関連しない |

## 機能種別

計算処理（テンソル演算）

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| filters | int | Yes（Noneも可） | 出力フィルタ数 | 正の整数 |
| kernel_size | int/tuple | Yes | 畳み込みウィンドウサイズ | rank次元のタプルに正規化 |
| strides | int/tuple | No（デフォルト1） | ストライド長 | dilation_rate!=1と排他 |
| padding | string | No（デフォルト"valid"） | パディング方式 | "valid"/"same"/"causal" |
| data_format | string | No | データ配置 | "channels_last"/"channels_first" |
| dilation_rate | int/tuple | No（デフォルト1） | ディレーション率 | strides!=1と排他 |
| groups | int | No（デフォルト1） | グループ数 | filters/groupsが整数 |
| activation | callable/string | No | 活性化関数 | 有効な活性化関数 |
| use_bias | bool | No（デフォルトTrue） | バイアス使用 | - |
| kernel_initializer | initializer | No（デフォルトglorot_uniform） | カーネル初期化 | 有効なinitializer |

### 入力データソース

前層からのテンソル出力。Conv2Dの場合: `(batch_size, height, width, channels)` または `(batch_size, channels, height, width)`。

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| output | tf.Tensor | 畳み込み結果テンソル。Conv2Dの場合: (batch, new_h, new_w, filters) |

### 出力先

次層への入力テンソル。

## 処理フロー

### 処理シーケンス

```
1. レイヤー初期化（__init__）
   └─ パラメータ正規化（kernel_size, strides, dilation_rate等）
2. ビルド（build）
   └─ カーネル重みとバイアス重みの作成
3. 順伝播（call）
   └─ nn_ops.convolutionによる畳み込み演算
   └─ バイアス加算
   └─ 活性化関数適用
```

### フローチャート

```mermaid
flowchart TD
    A[入力テンソル] --> B[Conv.call]
    B --> C{causalパディング?}
    C -->|Yes| D[入力のゼロパディング]
    C -->|No| E[パディング設定維持]
    D --> F[nn_ops.convolution]
    E --> F
    F --> G{use_bias?}
    G -->|Yes| H[nn.bias_add]
    G -->|No| I[スキップ]
    H --> J{activation?}
    I --> J
    J -->|Yes| K[活性化関数適用]
    J -->|No| L[出力]
    K --> L
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | ストライド・ディレーション排他 | strides!=1とdilation_rate!=1は同時指定不可 | Conv全般 |
| BR-02 | グループ畳み込み | input_channels/groupsがfilters/groupsのフィルタで畳み込み | groups > 1 |
| BR-03 | causalパディング | 1D畳み込みのみで有効。出力[t]は入力[t+1:]に依存しない | padding="causal" |
| BR-04 | channels_first制約 | channels_firstはGPUでのみ利用可能 | data_format設定時 |

### 計算ロジック

- 出力サイズ（same）: `output_size = ceil(input_size / stride)`
- 出力サイズ（valid）: `output_size = ceil((input_size - kernel_size + 1) / stride)`
- 有効受容野: `(kernel_size - 1) * dilation_rate + 1`

## データベース操作仕様

該当なし。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| ValueError | 負のfilters | filtersに負の値を指定 | 正の整数を指定 |
| ValueError | グループ不整合 | filtersまたはinput_channelsがgroupsで割り切れない | 整除可能な値を使用 |
| ValueError | ストライド・ディレーション競合 | 両方が1でない場合 | いずれかを1にする |

### リトライ仕様

該当なし。

## トランザクション仕様

該当なし。

## パフォーマンス要件

- GPU上ではcuDNNによる最適化された畳み込み実装を使用
- TensorFloat-32有効時はfloat32畳み込みが低精度高速化
- XLA JITコンパイル時はさらなる最適化が適用

## セキュリティ考慮事項

- 大きなカーネルサイズ・フィルタ数はメモリ枯渇の原因となり得る

## 備考

- Kerasの`Conv2D`は`tf.nn.conv2d`のラッパー
- `nn_ops.cc`にC++レベルのOp登録（Conv2D、Conv2DBackpropInput等）が定義

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | conv_utils.py | `tensorflow/python/keras/utils/conv_utils.py` | normalize_tuple、normalize_padding、conv_output_length |

**読解のコツ**: 畳み込みレイヤーのパラメータ（kernel_size、strides等）は内部でタプルに正規化される。`conv_utils.normalize_tuple`がその役割を担う。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | convolutional.py | `tensorflow/python/keras/layers/convolutional.py` | Convベースクラス（46行目〜） |

**主要処理フロー**:
1. **109-150行目**: __init__ - rank/filters/kernel_size/strides/padding等のパラメータ初期化
2. **build メソッド**: カーネル重み（kernel_shape）とバイアス重みの生成
3. **call メソッド**: nn_ops.convolution呼び出し、バイアス加算、活性化関数適用

#### Step 3: C++カーネル層を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | nn_ops.cc | `tensorflow/core/ops/nn_ops.cc` | Conv2D、Conv3D等のREGISTER_OP定義 |

**主要処理フロー**:
- **Conv2D Op**: input(T) + filter(T) -> output(T)、strides/padding/data_format属性
- **Conv2DBackpropInput**: 逆伝播用の入力勾配計算
- **Conv2DBackpropFilter**: 逆伝播用のフィルタ勾配計算

### プログラム呼び出し階層図

```
tf.keras.layers.Conv2D(filters, kernel_size, ...)
    |
    +-- Conv.__init__(rank=2, filters, kernel_size, ...)
    |       +-- conv_utils.normalize_tuple(kernel_size, 2, 'kernel_size')
    |       +-- conv_utils.normalize_padding(padding)
    |
    +-- Conv.build(input_shape)
    |       +-- self.add_weight('kernel', shape=kernel_shape)
    |       +-- self.add_weight('bias', shape=(filters,))
    |
    +-- Conv.call(inputs)
            +-- nn_ops.convolution(inputs, self.kernel, ...)
            |       +-- [C++] Conv2D Op (cuDNN on GPU)
            +-- nn.bias_add(outputs, self.bias)
            +-- self.activation(outputs)
```

### データフロー図

```
[入力]                    [処理]                         [出力]

入力テンソル         ---> nn_ops.convolution --------> 畳み込み結果
(batch,H,W,C)             |                           (batch,H',W',F)
                           v
カーネル重み          ---> カーネルとの相関計算
(kH,kW,C/g,F)             |
                           v
バイアス             ---> nn.bias_add
(F,)                       |
                           v
                      活性化関数適用
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| convolutional.py | `tensorflow/python/keras/layers/convolutional.py` | ソース | Keras畳み込みレイヤー実装 |
| nn_ops.cc | `tensorflow/core/ops/nn_ops.cc` | C++ソース | Conv2D等のOp定義 |
| nn_ops.py | `tensorflow/python/ops/nn_ops.py` | ソース | nn_ops.convolution Python実装 |
| nn.py | `tensorflow/python/ops/nn.py` | ソース | tf.nn名前空間 |
| conv_utils.py | `tensorflow/python/keras/utils/conv_utils.py` | ソース | パラメータ正規化ユーティリティ |
| base_layer.py | `tensorflow/python/keras/engine/base_layer.py` | ソース | Layerベースクラス |
